const { Resend } = require("resend");
const fs = require("fs");
const path = require("path");

// Try to load .env.local or .env file
function loadEnv() {
  const envFiles = [".env.local", ".env"];
  for (const envFile of envFiles) {
    const envPath = path.join(__dirname, "..", envFile);
    if (fs.existsSync(envPath)) {
      const envContent = fs.readFileSync(envPath, "utf8");
      envContent.split("\n").forEach((line) => {
        const match = line.match(/^([^=:#]+)=(.*)$/);
        if (match) {
          const key = match[1].trim();
          const value = match[2].trim().replace(/^["']|["']$/g, "");
          if (!process.env[key]) {
            process.env[key] = value;
          }
        }
      });
      break;
    }
  }
}

loadEnv();

const EMAIL_FROM = process.env.EMAIL_FROM || "noreply@hospital.com";
const EMAIL_FROM_NAME = process.env.EMAIL_FROM_NAME || "Hospital Task System";
const TEST_EMAIL = "shayan.khalighi76@gmail.com";

async function testEmail() {
  console.log("📧 Testing email service...");
  console.log(`From: ${EMAIL_FROM_NAME} <${EMAIL_FROM}>`);
  console.log(`To: ${TEST_EMAIL}`);
  console.log(
    `API Key: ${process.env.RESEND_API_KEY ? "✅ Set" : "❌ Not set"}`
  );
  console.log("");

  if (!process.env.RESEND_API_KEY) {
    console.error("❌ RESEND_API_KEY is not set in environment variables");
    console.log("");
    console.log("Please add RESEND_API_KEY to your .env.local file:");
    console.log("RESEND_API_KEY=re_xxxxxxxxxxxxxxxxxxxxx");
    console.log("");
    console.log("Or set it as an environment variable:");
    console.log("RESEND_API_KEY=re_xxx node scripts/test-email.js");
    process.exit(1);
  }

  const resend = new Resend(process.env.RESEND_API_KEY);

  try {
    // Test 1: Simple email
    console.log("Test 1: Sending simple test email...");
    const { data, error } = await resend.emails.send({
      from: `${EMAIL_FROM_NAME} <${EMAIL_FROM}>`,
      to: [TEST_EMAIL],
      subject: "Test Email - Hospital Task System",
      html: `
        <!DOCTYPE html>
        <html>
          <head>
            <meta charset="utf-8">
            <style>
              body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
              .container { max-width: 600px; margin: 0 auto; padding: 20px; }
              .header { background-color: #2563eb; color: white; padding: 20px; border-radius: 8px 8px 0 0; }
              .content { background-color: #f9fafb; padding: 20px; border-radius: 0 0 8px 8px; }
              .button { display: inline-block; padding: 12px 24px; background-color: #2563eb; color: white; text-decoration: none; border-radius: 6px; margin-top: 15px; }
            </style>
          </head>
          <body>
            <div class="container">
              <div class="header">
                <h1>✅ Email Service Test</h1>
              </div>
              <div class="content">
                <p>Hello!</p>
                <p>This is a test email from the Hospital Task System.</p>
                <p>If you received this email, the email service is working correctly! 🎉</p>
                <p><strong>Test Details:</strong></p>
                <ul>
                  <li>Service: Resend</li>
                  <li>From: ${EMAIL_FROM_NAME} <${EMAIL_FROM}></li>
                  <li>Time: ${new Date().toLocaleString()}</li>
                </ul>
                <p>You can now use the email notification system for tasks and tickets.</p>
              </div>
            </div>
          </body>
        </html>
      `,
    });

    if (error) {
      console.error("❌ Error sending email:", error);
      process.exit(1);
    }

    console.log("✅ Email sent successfully!");
    console.log("Email ID:", data?.id);
    console.log("");
    console.log("📬 Please check your inbox at:", TEST_EMAIL);
    console.log("(Also check spam folder if you don't see it)");

    // Test 2: Task assignment email
    console.log("");
    console.log("Test 2: Sending task assignment email...");
    const taskEmail = await resend.emails.send({
      from: `${EMAIL_FROM_NAME} <${EMAIL_FROM}>`,
      to: [TEST_EMAIL],
      subject: "Test: New Task Assigned - Sample Task",
      html: `
        <!DOCTYPE html>
        <html>
          <head>
            <meta charset="utf-8">
            <style>
              body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
              .container { max-width: 600px; margin: 0 auto; padding: 20px; }
              .header { background-color: #2563eb; color: white; padding: 20px; border-radius: 8px 8px 0 0; }
              .content { background-color: #f9fafb; padding: 20px; border-radius: 0 0 8px 8px; }
              .task-card { background-color: white; padding: 15px; margin: 15px 0; border-left: 4px solid #2563eb; border-radius: 4px; }
              .button { display: inline-block; padding: 12px 24px; background-color: #2563eb; color: white; text-decoration: none; border-radius: 6px; margin-top: 15px; }
            </style>
          </head>
          <body>
            <div class="container">
              <div class="header">
                <h1>New Task Assigned</h1>
              </div>
              <div class="content">
                <p>Hello,</p>
                <p>You have been assigned a new task:</p>
                <div class="task-card">
                  <h2 style="margin-top: 0;">Sample Test Task</h2>
                  <p><strong>Description:</strong></p>
                  <p>This is a test task to verify the email notification system is working correctly.</p>
                  <p><strong>Due Date:</strong> ${new Date(
                    Date.now() + 7 * 24 * 60 * 60 * 1000
                  ).toLocaleDateString()}</p>
                  <p><strong>Assigned by:</strong> Test Admin (admin@hospital.com)</p>
                </div>
                <a href="http://localhost:3000/tasks/test-task-id" class="button">View Task</a>
              </div>
            </div>
          </body>
        </html>
      `,
    });

    if (taskEmail.error) {
      console.error("❌ Error sending task assignment email:", taskEmail.error);
    } else {
      console.log("✅ Task assignment email sent successfully!");
      console.log("Email ID:", taskEmail.data?.id);
    }

    console.log("");
    console.log("✨ All tests completed!");
  } catch (error) {
    console.error("❌ Failed to send email:", error);
    process.exit(1);
  }
}

testEmail();
